#!/usr/bin/python3
# -*- coding: utf8 -*-
""" Parse Conf Files """
from __future__ import print_function
import time

# Multi-language
import multilang_tools
import ml_sched_parser
T = multilang_tools.ml_server(ml_sched_parser.S)

def set_schedp_lang(lang):
    global T
    T = multilang_tools.ml_server(ml_sched_parser.S, lang)

def valid_sched(sched_toks,sched30):
    """ Validate and compile a schedule line """
    sec_in_a_day = 24*60*60
    sched_type = sched_toks[0].lower()
    msg = None
    try:
        if   sched_type == "daily":     # daily
            fmt_str = "%H:%M"
            fmt2_str = "%H:%M"
        elif sched_type == "weekly":    # weekly
            fmt_str = "%a %H:%M"
            fmt2_str = "%d %H:%M"
        elif sched_type == "monthly":   # monthly
            fmt_str = "%d %H:%M"
            fmt2_str = "%d %H:%M"
        else:   # Othre (error)
            return False, T.m000%sched_toks[0]

        if sched_toks[2].startswith('+'):   # Repeat
            start = time.strptime(sched_toks[1],fmt_str)
            rpeat = sched_toks[2][1:].split('*')

            # Validate
            if sched_type == "monthly":
                msg = T.m001
                raise ValueError()
            if len(rpeat) != 2:
                msg = T.m002%sched_toks[2]
                raise ValueError()
            intvl = time.strptime(rpeat[0],fmt2_str)
            num = -1 if rpeat[1] == "?" else int(rpeat[1] )
            if num == 0:
                msg = T.m003
                raise ValueError()
            if num < 0:
                num = (24 * 2 if sched30 else 24 * 60)
                num *= (7 if sched_type == "weekly" else 1)
            if sched30 and start.tm_min%30:
                msg = T.m004%sched_toks[1]
                raise ValueError()
            if sched30 and intvl.tm_min%30:
                msg = T.m004%rpeat[0]
                raise ValueError()

            # Compile (final values are in UNIX secs)
            sched_form = "repeat"
            tt_whole = sec_in_a_day
            tt_start = 60*(start.tm_min + 60*start.tm_hour)
            tt_intvl = 60*(intvl.tm_min + 60*intvl.tm_hour)
            if sched_type == "weekly":
                tt_whole *= 7
                tt_start += sec_in_a_day * start.tm_wday
                tt_intvl += sec_in_a_day * intvl.tm_wday
            if tt_intvl == 0:
                msg = T.m005
                raise ValueError()

            # Handle continuous mode
            cont_flag = False
            cont_thrsh = 3.6
            if sched30:
                if len(sched_toks) > 3:
                    for t in sched_toks[3:]:
                        if t:
                            msg = T.m006
                            raise ValueError()
            else:
                try:
                    last_val = float(sched_toks[-1] )
                    if last_val <= 5.0 and last_val >=0.0:
                        cont_flag = True
                        cont_thrsh = last_val
                except ValueError:
                    pass

            return True, {
                "sched_type":sched_type,
                "sched_form":sched_form,
                "sched":(tt_start,tt_intvl,num,tt_whole),
                "cont_flag":cont_flag,
                "cont_thrsh":cont_thrsh,
                }
        else:
            sched_form = "definite"
            sched_wk = []
            sched_tokens = []
            for i in sched_toks[1:]:
                if i:
                    # Validate
                    tm = time.strptime(i,fmt_str)
                    if sched30 and tm.tm_min%30:
                        msg = T.m004%i
                        raise ValueError()
                    # Compile (final values are in UNIX secs)
                    tt_sched = 60*(tm.tm_min + 60*tm.tm_hour)
                    if sched_type == "weekly": tt_sched += sec_in_a_day * tm.tm_wday
                    if sched_type == "monthly": tt_sched += sec_in_a_day * (tm.tm_mday - 1)
                    sched_wk.append(tt_sched)
            # Validate for entries
            if not sched_wk:
                msg = T.m007
                raise ValueError()
            # sort and remove duplicate
            sched_wk.sort()
            prev = -1
            for i in sched_wk:
                if i != prev:
                    sched_tokens.append(i)
                prev = i
            return True, {"sched_type":sched_type,"sched_form":sched_form,"sched":tuple(sched_tokens) }

    except Exception as e:
        if type(e) is ValueError and msg is not None:
            return False, msg
        else:
            return False, repr(e)
    return False, "Thsi should not happen"

def parse_sched_line(sched,now,offset=0):
    """ Determine schedule in UNIX time for "now" and "next" """
    def find_now(p,now):
        # Find the schedule for now
        abs_diff = []
        for i in p:
            abs_diff.append(abs(i - now) )
        idx = abs_diff.index(min(abs_diff) )
        return p[idx],idx

    ### Design Basisi for "repeat" schedule
    #
    ## Definitions
    # whole = whole period of schedule unit (day, week or month)
    # elapsed = now - bgn  --->  now = bgn + elapsed
    # shced(n) = bgn + start + n * intvl
    #
    ## Largest possible n and corresponding time
    # Nh = floor((whole - start) / intvl) - 1
    # Th = start + Nh * intvl
    #
    ## Largest n for elapsed > start + n * intvl
    # Nm = floor((elapsed - start) / intvl)
    #
    ## Smallest n for elapsed < start + n * intvl
    # Ns = ceil((elapsed - start) / intvl)
    #
    ## If elapsed < start, compare following A and B, and take the event that gives the smaller
    # A = start - elapsed            [first schedule of this unit]
    # B = elapsed - (Th - whole)     [last schedule of previous unit]
    #
    ## If elapsed > Th, compare following A and B, and take the event that gives the smaller
    # A = elapsed - Th               [last schedule of this unit]
    # B = (start + whole) - elapsed  [first schedule of next unit]
    
    # beginning of the schedule unit for daily and weekly
    str_today = time.strftime("%Y%m%d",time.localtime(now) )
    tm_today = time.strptime(str_today,"%Y%m%d")
    tt_today = time.mktime(tm_today)
    tt_last_monday = tt_today - 24*60*60*tm_today.tm_wday
    if   sched["sched_type"] == "daily":
        tt_bgn = tt_today
    elif sched["sched_type"] == "weekly":
        tt_bgn = tt_last_monday

    # Schedule forms
    if   sched["sched_form"] == "repeat":
        if sched["sched_type"] not in "daily weekly".split():
            raise ValueError("Invalid schedule type \"%s\" for repeat form"%sched["sched_type"] )

        dd = sched["sched"]
        start = dd[0]
        intvl = dd[1]
        num   = dd[2]
        whole = dd[3]

        elapsed = now - tt_bgn - offset
        # nh = min(int((whole - start ) / intvl ) - 1, num - 1)
        nh = min(int((whole - start ) / intvl ), num)
        th = start + nh * intvl
        if th >= whole:
            nh -= 1
            th -= intvl
        # print("num = %d, nh = %d, th = %d = %02d:%02d"%(num, nh,th,int(th/3600), int(th/60)%60) )
        nm = int((elapsed - start ) / intvl )
        ns = nm + 1

        if   elapsed >= start and elapsed <= th:    # Normal case
            tt_m = start + nm * intvl
            tt_s = start + ns * intvl
        elif elapsed < start:                       # before start of repeating schedule
            tt_m = th - whole
            tt_s = start
        elif elapsed > th:                          # beyond end of repeating schedule
            tt_m = th
            tt_s = start + whole

        # Choose closer schedule
        if  (elapsed - tt_m) < (tt_s - elapsed):
            tt_now = tt_m + tt_bgn
            tt_nxt = tt_s + tt_bgn
        else:
            tt_now = tt_s + tt_bgn
            tt_nxt = tt_s + intvl
            if tt_nxt > (th +whole ):   # Beyond upper limit of next schedule unit
                tt_nxt = start + 2*whole
            elif tt_nxt > th and tt_nxt < (start + whole ): # Between upper limit and start of next schedule unit
                tt_nxt = start + whole  # start of next unit
            tt_nxt += tt_bgn

    elif sched["sched_form"] == "definite":
        if   sched["sched_type"] in "daily weekly".split():
            w = 24*60*60 if sched["sched_type"] == "daily" else 7*24*60*60
            s = list(sched["sched"] )
            p = [s[-1] - w,] + s + [i+w for i in s] + [s[0] + 2*w,]
        elif sched["sched_type"] == "monthly":
            # Calculate tt for beginnig of this month
            str_this_mo = time.strftime("%Y%m",time.localtime(now) )
            tm_this_mo = time.strptime(str_this_mo,"%Y%m")
            tt_this_mo = time.mktime(tm_this_mo)
            # Calculate tt for beginnig of previous month
            prev_mo,prev_yr = ((tm_this_mo.tm_mon-1,tm_this_mo.tm_year)
                                if tm_this_mo.tm_mon != 1 else
                               (12,tm_this_mo.tm_year-1) )
            tm_prev_mo = time.strptime("%d%02d"%(prev_yr,prev_mo),"%Y%m")
            tt_prev_mo = time.mktime(tm_prev_mo)
            # Calculate tt for beginnig of next month
            next_mo,next_yr = ((tm_this_mo.tm_mon+1,tm_this_mo.tm_year)
                                if tm_this_mo.tm_mon != 12 else
                               (1,tm_this_mo.tm_year+1) )
            tm_next_mo = time.strptime("%d%02d"%(next_yr,next_mo),"%Y%m")
            tt_next_mo = time.mktime(tm_next_mo)
            # Calculate tt for beginnig of the month after next
            tman_mo,tman_yr = ((tm_this_mo.tm_mon+2,tm_this_mo.tm_year)
                                if tm_this_mo.tm_mon < 11 else
                               (tm_this_mo.tm_mon - 10,tm_this_mo.tm_year+1) )
            tm_tman_mo = time.strptime("%d%02d"%(tman_yr,tman_mo),"%Y%m")
            tt_tman_mo = time.mktime(tm_tman_mo)

            tt_bgn = tt_this_mo
            tt_w_prev = tt_this_mo - tt_prev_mo
            tt_w_this = tt_next_mo - tt_this_mo
            tt_w_next = tt_tman_mo - tt_next_mo
            prt_tt(tt_prev_mo)
            prt_tt(tt_this_mo)
            prt_tt(tt_next_mo)
            prt_tt(tt_tman_mo)
            print(tt_w_prev/24./60/60,tt_w_this/24./60/60,tt_w_next/24./60/60)
            s_prev = [i-tt_w_prev for i in sched["sched"] if i < tt_w_prev]
            s_this = [i           for i in sched["sched"] if i < tt_w_this]
            s_next = [i+tt_w_this for i in sched["sched"] if i < tt_w_next]
            p = [s_prev[-1],] + s_this + s_next + [s_next[0]+tt_w_next,]
        else: raise ValueError("Invalid schedule type \"%s\""%sched["sched_type"] )
        e = now - tt_bgn - offset
        tt_now,idx = find_now(p,e)
        tt_now += tt_bgn
        tt_nxt = p[idx + 1] + tt_bgn
    else: raise ValueError("Invalid schedule form \"%s\""%sched["sched_form"] )

    return tt_now+offset, tt_nxt+offset

def mk_strftime(tt):
    return time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(tt) )

def prt_tt(tt):
    print(mk_strftime(tt) )

if __name__ == "__main__":
    text = """# Sample schedule file
error_no_entry, daily, , , , , , , , , ,
error_not_30, daily, 00:00,+00:01*?, , , , , , , ,
error_bad_type, annual, 25 09:30, 15 09:30, 25 09:30, 05 09:30, , , , , ,
error_bad_day, weekly, Mon 09:30, tue 09:30, wed 09:30, san 09:30, , , , , ,
error_bad_day, monthly, 35 09:30, 15 09:30, 25 09:30, 05 09:30, , , , , ,
error_missing_day,  weekly, Mon 00:00,+02:30*1000, , , , , , , ,
error_monthly_repeat,  monthly, 01 00:00, +00 00:30*?, , , , , , , ,
daily definite, daily, 06:30, 07:30, , , , , , , ,
weekly definite, weekly, mon 06:30, sun 07:30, , , , , , , ,
monthly definite, monthly, 28 23:30, 29 23:30, 30 23:30, 31 23:30, 28 00:00, , , , ,
#P-456, weekly, Mon 09:30, Wed 09:30, Fri 09:30, , , , , , ,
# C-567, monthly, Mon1 09:30, Mon2 09:30, Mon3 09:30, Mon4 09:30, , , , , ,
# C-657, monthly, 05 09:30, 15 09:30, 25 09:30, , , , , , ,
daily repeat,  daily, 00:00, +00:30*3, , , , , , , ,
daily repeat, daily, 09:30, +00:30*?, , , , , , , ,
proto2, daily, 00:00, +05:00*?, , , , , , , ,
"""

    offset = 0
    now = time.mktime(time.strptime("20180709 150605", "%Y%m%d %H%M%S") )
    lines = [i.split("#")[0].strip() for i in text.splitlines() ]
    lines = [i for i in lines if i]
    for l in lines:
        print(l)
        toks = [i.strip() for i in l.split(",") ]
        a,b = valid_sched(toks[1:], True)
        if a:
            print("\tOK: ",b)
            result = [mk_strftime(i) for i in parse_sched_line(b, now, offset) ]
            print("\tOK :",result)
            assert not l.startswith("error"),"Should not happen"
        else:
            print("\tError: "+b.encode("sjis") )
            assert l.startswith("error"),"Should not happen"
