#!/usr/bin/python3
# -*- coding: utf8 -*-
""" Parse Conf Files """
from __future__ import print_function
import codecs,hashlib,os,pickle,sys,struct,threading,time,traceback,warnings
from sched_parser import valid_sched, parse_sched_line, set_schedp_lang
import platform
if platform.platform().lower().startswith("windows-10"):
    import ctypes
    kernel32 = ctypes.windll.kernel32
    kernel32.SetConsoleMode(kernel32.GetStdHandle(-11), 7)
    esc_norm = "\x1b[0m"
    esc_strng = "\x1b[101;93m"
    esc_grbg = "\x1b[42;97m"
    esc_blfg = "\x1b[47;94m"
    esc_nega = "\x1b[107;30m"
    esc_neg2 = "\x1b[47;30m"
else:
    esc_norm = ""
    esc_strng = ""
    esc_grbg = ""
    esc_blfg = ""
    esc_nega = ""
    esc_neg2 = ""

# Private modules
from cna_bts_version import BTS_VERSION
REPORT_TITLE = "Generated by conanair Base Station App. "+BTS_VERSION + " (c) NSXe, All rights reserved.\n\n"
CAIR_ROOT = "CAIR_ROOT"
UTF8BOM = b'\xef\xbb\xbf'.decode("utf-8")
PRIV_DIR = "private"

# Multi-language
import multilang_tools
import ml_conf_parser
T = multilang_tools.ml_server(ml_conf_parser.S)

def set_confp_lang(lang):
    global T
    T = multilang_tools.ml_server(ml_conf_parser.S, lang)
    set_schedp_lang(lang)

class key_tag_sched(object):
    def __init__(self):
        self.last_conf_load = 0
        self.reported_missing = []
        self.prev_dsec = {}

        # picles
        pkl_dir = os.path.join("CAIR_ROOT", PRIV_DIR)
        pkl_path = os.path.join(pkl_dir,"stat_history.pkl")
        if not os.path.exists(pkl_dir): os.makedirs(pkl_dir)
        self.STAT_PICKLE = pkl_path
        self.STAT_LOCK = threading.Lock()

        # init
        self.load_only()
        self.init_stat()
        try:
            with codecs.open(self.rpt_latest_sched,"w", encoding='utf-8', errors='backslashreplace') as f:
                self.config_report(f)
        except IOError as e:
            if (not self.aprt.is_verbose() ) and (e[0] == 13): pass
            else: raise

        # PID
        apprt = self.aprt.get_root()
        if not os.path.exists(apprt): os.makedirs(apprt)
        try:
            with codecs.open(os.path.join(pkl_dir,"pid"),"w", encoding='utf-8', errors='backslashreplace') as f:    # CAIR_ROOT
                f.write("%d\n"%os.getpid() )
            with codecs.open(os.path.join(apprt,"pid"),"w", encoding='utf-8', errors='backslashreplace') as f:      # APP_ROOT
                f.write("%d\n"%os.getpid() )
            with codecs.open(self.rpt_params,"w", encoding='utf-8', errors='backslashreplace') as f:
                self.param_report(f)
        except IOError as e:
            if (not self.aprt.is_verbose() ) and (e[0] == 13): pass
            else: raise

    def init_stat(self):
        if os.path.exists(self.STAT_PICKLE):
            try:
                # Read pickle
                with open(self.STAT_PICKLE, "rb") as f:
                    pkl_data = f.read()
                pkl_hash = pkl_data[:16]
                pkl_data = pkl_data[16:]
                # check hash
                m = hashlib.md5()
                m.update(pkl_data)
                new_hash = m.digest()
                stat = {}
                if pkl_hash == new_hash:
                    pkl_dict = pickle.loads(pkl_data)
                    if (    "data" in pkl_dict.keys()
                        and "init" in pkl_dict.keys()
                        and "last" in pkl_dict.keys() ):
                        stat["init"] =  pkl_dict["init"]
                        stat["last"] =  pkl_dict["last"]
                        stat["data"] = {}
                        for i in pkl_dict["data"].keys():
                            stat["data"][i] = pkl_dict["data"][i]
                        self.stat_dict = stat
                        self.updt_stat_rcds()
                        return
            except Exception as e:
                print(esc_strng+"========= Error in reading pickle =========",file=sys.stderr)
                traceback.print_exc()
                print("=========== pickled data ignored =========="+esc_norm,file=sys.stderr)

        # No good pickle available
        tag_id = self.ids.items()           # list of (tag,id)
        stat = {}
        stat["data"] = {}
        now = time.time()
        str_now = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(now) )
        stat["init"] =  {"ts_str":str_now,"ts_num":now}
        stat["last"] =  {"ts_str":"None","ts_num":0}
        for i in tag_id:
            # "data":{"tag": {"id":, "last":{"ts_str":,"ts_num":,"schd_str":,"schd_num":,"vbat":}, "next":{"ts_str":,"ts_num":} } }
            stat["data"][i[0] ] = {
                    "id":i[1],
                    "last":{"schd_str":"None","schd_num":0,"ts_str":"None","ts_num":0,"vbat":0},
                    "next":{"schd_str":"None","schd_num":0}
                }
        self.stat_dict = stat

    def updt_stat_rcds(self):
        stat = self.stat_dict
        tag_id = self.ids.items()           # list of (tag,id)
        # Add new data
        for i in tag_id:    # i = (tag,id)
            # tag: {"id":, "last":{"ts_str":,"ts_num":,"vbat":}, "next":{"ts_str":,"ts_num":} }
            if i[0] in self.stat_dict["data"].keys():
                stat["data"][i[0] ]["id"] = i[1]
            else:
                stat["data"][i[0] ] = {
                        "id":i[1],
                        "last":{"schd_str":"None","schd_num":0,"ts_str":"None","ts_num":0,"vbat":0},
                        "next":{"schd_str":"None","schd_num":0}
                    }
        # Remove data no longer exist
        i_to_del = []
        for i in stat["data"].keys():
            if i not in self.ids.keys():
                i_to_del.append(i)
        for i in i_to_del:
            del stat["data"][i]
        self.stat_dict = stat

    def update_stat_by_tag(self,tag,vbat,last_tt,now_tt,next_tt):
        if tag not in self.stat_dict["data"].keys(): raise ValueError("No stat record for tag %s"%tag)
        str_tt = lambda x: time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(x) )
        # Acquire lock and update
        self.STAT_LOCK.acquire()
        self.stat_dict["data"][tag]["last"]["schd_str"] = str_tt(now_tt)
        self.stat_dict["data"][tag]["last"]["schd_num"] = now_tt
        self.stat_dict["data"][tag]["last"]["ts_str"] = str_tt(last_tt)
        self.stat_dict["data"][tag]["last"]["ts_num"] = last_tt
        self.stat_dict["data"][tag]["last"]["vbat"] =   vbat
        self.stat_dict["data"][tag]["next"]["schd_str"] = str_tt(next_tt)
        self.stat_dict["data"][tag]["next"]["schd_num"] = next_tt
        self.stat_dict["last"]["ts_str"] = str_tt(last_tt)
        self.stat_dict["last"]["ts_num"] = last_tt
        self.STAT_LOCK.release()
        try:
            self.reported_missing.remove(tag)
        except ValueError:
            pass
        try:
            del self.prev_dsec[tag]
        except KeyError:
            pass

    def update_stat_by_id(self,id,vbat,last_tt,next_tt):
        tag = self.get_tag(id)
        if tag is not None:
            if self.stat_dict["data"][tag]["id"] == id:
                self.update_stat_by_tag(tag,vbat,last_tt,next_tt)
            else:
                raise ValueError("ID unmatch: %s vs. %s"%(id,self.stat_dict["data"][tag]["id"] ) )
        else:
            raise ValueError("No tag available for ID %s"%id)

    def param_report(self,f=None):
        now = round(time.time() )
        str_now = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(now) )

        if f is None or f is sys.stdout:
            # enc = "sjis"
            pre = esc_neg2
            post = esc_norm
            rpt = esc_neg2 + REPORT_TITLE[:-2] + esc_norm + "\n\n"
        else:
            # enc = "utf8"
            pre = ""
            post = ""
            rpt = UTF8BOM + REPORT_TITLE
        non_verbose = "app_root data_gen data_sync file_type summary_csv sched30 sched_offset".split()
        param,default = self.aprt.get_pdict()
        keys = list(set(list(param.keys() ) + list(default.keys() ) ) )
        rpt_type = ""
        if not self.aprt.is_verbose():
            # rpt_type = pre+T.m000+post
            keys = [i for i in keys if i in non_verbose]
        keys.sort()

        rpt += T.m001a%str_now+rpt_type+T.m001b
        rpt += T.m002%time.strftime("%Y-%m-%d %H:%M:%S",
                                                   time.localtime(self.last_conf_load) )
        rpt += T.m003
        for k in keys:
            v = param.get(k)

            # No summary_csv listing if default is used for it
            if k == "summary_csv" and v is None: continue

            if v is None:
                d = "Yes"
                v = default[k]
            else:
                d = "No"

            rpt += "%-20s"%k
            rpt += "%-30s %s\n"%(str(v),d)

            # Effective file type
            if k == "summary_csv":
                rpt += (15*" " + "(effective file type = %s)\n"%self.aprt.get_effective_file_type() )

        if f is None:
            return rpt
        else:
            # f.write(rpt.encode(enc) )
            f.write(rpt)
        if f is sys.stdout: f.flush()

    def stat_report(self,f=None,bad_only=False):
        def h_m_s(sec):
            ss = int(sec)
            h = int(ss / 3600)
            m = int(ss / 60) % 60
            s = ss % 60
            if   abs(sec) < 60:
                return "%ds"%s
            elif abs(sec) < 3600:
                return "%dm%3ds"%(m,s)
            return "%dh%3dm%3ds"%(h,m,s)

        # Start Report
        self.STAT_LOCK.acquire()
        now = round(time.time() )
        str_now = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(now) )

        if f is None or f is sys.stdout:
            # enc = "sjis"
            pre = esc_strng
            post = esc_norm
            rpt = esc_neg2 + REPORT_TITLE[:-2] + esc_norm + "\n\n"
        else:
            # enc = "utf8"
            pre = ""
            post = ""
            rpt = UTF8BOM + REPORT_TITLE

        rpt += T.m004%str_now
        rpt += (T.m005
                    %(self.stat_dict["init"]["ts_str"],self.stat_dict["last"]["ts_str"] ) )
        if bad_only: rpt += esc_strng+T.m006+esc_norm+"\n" 
        rpt += T.m007

        dict = self.stat_dict["data"]
        tags = [i for i in dict.keys() ]
        tags.sort()
        cnt = 0
        for i in tags:
            dd = dict[i]
            rmk = ""
            bad = False
            if dd["next"]["schd_num"] > 0:
                dsec = now - dd["next"]["schd_num"]
                bad = dsec > 60
                rmk = ((h_m_s(-dsec) if dsec < 0 else
                    (T.m008a if dsec < 60 else pre+T.m008b%h_m_s(dsec) )
                    +("" if dsec < 60 else (post if dsec < 90 else 
                        ("!"+post if dsec < 300 else ("!!"+post if dsec < 900 else "!!!"+post) ) ) ) ) )
            if (not bad_only) or bad:
                cnt += 1
                # tag, id, last data time, last schedule, Vbat, next schedule, remarks
                rpt += ("%-16s%-12s%-21s%-21s %6.3f  %-21s %s\n"
                    %(i,dd["id"],
                      dd["last"]["ts_str"],dd["last"]["schd_str"],dd["last"]["vbat"],
                      dd["next"]["schd_str"],rmk) )
        if bad_only and cnt == 0:
            rpt += "            "+esc_grbg+T.m009+esc_norm+"\n"
        rpt += T.m010
        rpt += T.m011
        rpt += T.m012
        rpt += T.m013

        self.STAT_LOCK.release()

        # Save pickle
        self.save_stat()

        if f is None:
            return rpt
        else:
            # f.write(rpt.encode(enc) )
            f.write(rpt)
        if f is sys.stdout: f.flush()

    def save_stat(self):
        # Make pickle
        self.STAT_LOCK.acquire()
        pkl_data = pickle.dumps(self.stat_dict)
        self.STAT_LOCK.release()

        # generate hash
        m = hashlib.md5()
        m.update(pkl_data)
        pkl_hash = m.digest()

        # write pickle
        try:
            with open(self.STAT_PICKLE, "wb") as p:
                p.write(pkl_hash+pkl_data)
        except IOError as e:
            if (not self.aprt.is_verbose() ):
                print(esc_strng+"========= Error in writing pickle =========",file=sys.stderr)
                traceback.print_exc()
                print("========= Error in writing pickle ========="+esc_norm,file=sys.stderr)
            else: raise

    def config_report(self,f=None):
        now = time.time()
        str_now = time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(now) )

        if f is None or f is sys.stdout:
            # enc = "sjis"
            pre = esc_strng
            post = esc_norm
            rpt = esc_neg2 + REPORT_TITLE[:-2] + esc_norm+"\n\n"
        else:
            # enc = "utf8"
            pre = ""
            post = ""
            rpt = UTF8BOM + REPORT_TITLE

        rpt += T.m014%str_now
        rpt += T.m015%time.strftime("%Y-%m-%d %H:%M:%S",
                                                   time.localtime(self.last_conf_load) )
        rpt += T.m016

        # Default Schedule
        rpt += "*               (Default)            \"%s\"\n"%self.txt_def_sched
        
        tags = [i for i in self.ids.keys() ]
        tags.sort()
        for t in tags:
            i = self.ids[t]
            s = self.txt_sched.get(i)
            s = T.m017 if s is None else s
            o = "%+4d"%self.offset_from_tag[t]
            rpt += T.m018%(t,i,o,s)
        if f is None:
            return rpt
        else:
            # f.write(rpt.encode(enc) )
            f.write(rpt)
        if f is sys.stdout: f.flush()

    def load(self):
        self.load_only()
        self.updt_stat_rcds()
        try:
            with codecs.open(self.rpt_latest_sched,"w", encoding='utf-8', errors='backslashreplace') as f:
                self.config_report(f)
            with codecs.open(self.rpt_params,"w", encoding='utf-8', errors='backslashreplace') as f:
                self.param_report(f)
        except IOError as e:
            if (not self.aprt.is_verbose() ) and (e[0] == 13): pass
            else: raise

    def load_only(self):
        self.key_dir = os.path.join(CAIR_ROOT,"connection_keys")
        self.def_sched = None

        # APP_ROOT
        self.aprt = app_root_class()
        VERBOSE = self.aprt.is_verbose()

        # Report Files
        self.rpt_key_error =    os.path.join(self.aprt.get_root(), "key_error.txt")
        self.rpt_tags_error =   os.path.join(self.aprt.get_root(), "tags_error.txt")
        self.rpt_sched_error =  os.path.join(self.aprt.get_root(), "sched_error.txt")
        self.rpt_key_vs_tags =  os.path.join(self.aprt.get_root(), "key_vs_tags.txt")
        self.rpt_latest_sched = os.path.join(self.aprt.get_root(), "schedule.txt")
        self.rpt_measurements = os.path.join(self.aprt.get_root(), "status.txt")
        self.rpt_evtlog =       os.path.join(self.aprt.get_root(), "evtlog.txt")
        self.rpt_params =       os.path.join(self.aprt.get_root(), "appparam.txt")

        _keyfile = {}
        _keys = {}
        _tags = {}
        _ids = {}

        # Connection Keys
        try:
            with codecs.open(self.rpt_key_error, "w", encoding='utf-8', errors='backslashreplace') as f:
                f.write(UTF8BOM)    # Clear report file
                f.write(REPORT_TITLE)

            keyfile_list = [i for i in os.listdir(self.key_dir)
                                if i.lower().endswith(".key")
                                and os.path.isfile(os.path.join(self.key_dir,i) ) ]
            for k in keyfile_list:
                key_path = os.path.join(self.key_dir,k)
                with open(key_path, "rb") as f:
                    key_data = f.read()
                if len(key_data) != 80:
                    with codecs.open(self.rpt_key_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m019%k)
                    continue
                id = k.split('.')[0]
                if id != key_data[:len(id) ].decode():
                    with codecs.open(self.rpt_key_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m020%k )
                    continue
                _keys[id] = key_data[16:]
                _keyfile[id] = k
        except IOError as e:
            if (not self.aprt.is_verbose() ) and (e[0] == 13): pass
            else: raise

        # tags
        try:
            with codecs.open(self.rpt_tags_error, "w", encoding='utf-8', errors='backslashreplace') as f:
                f.write(UTF8BOM)    # Clear report file
                f.write(REPORT_TITLE)
            with codecs.open(os.path.join(CAIR_ROOT,"tags.conf"),"r", encoding='utf-8', errors='backslashreplace') as f:
                tag_data = f.read()
                if tag_data[0] == UTF8BOM: tag_data = tag_data[1:]
            tag_lines = [i.split('#')[0].strip() for i in tag_data.splitlines() ]
            taglist = []
            idlist = []
            for l,num in zip(tag_lines,range(len(tag_lines) ) ):
                if not l or l == UTF8BOM: continue

                tags_err = False
                tokens = l.split(",")
                if len(tokens) < 2:
                    with codecs.open(self.rpt_tags_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m021%(num+1) )
                    continue
                if len(tokens) > 2:
                    with codecs.open(self.rpt_tags_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m022%(num+1)  )
                t,i = tokens[:2]
                t = t.strip()
                i = i.strip()
                if t in taglist:
                    with codecs.open(self.rpt_tags_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m023%(num+1) )
                    tags_err = True
                if i in idlist:
                    with codecs.open(self.rpt_tags_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m024%(num+1) )
                    tags_err = True
                if tags_err: continue

                taglist.append(t)
                idlist.append(i)
                _tags[i] = t
                _ids[t] = i
        except IOError as e:
            if (not self.aprt.is_verbose() ) and (e[0] == 13): pass
            else: raise

        # check for missing tag but key
        try:
            with codecs.open(self.rpt_key_vs_tags, "w", encoding='utf-8', errors='backslashreplace') as f:
                f.write(UTF8BOM)    # Clear report file
                f.write(REPORT_TITLE)
            k_id_to_del = []
            for k_id in _keyfile.keys():
                if k_id not in _tags.keys():
                    with codecs.open(self.rpt_key_vs_tags, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m025%_keyfile[k_id] )
                    # keyfile_list.remove(_keyfile[k_id] )
                    k_id_to_del.append(k_id)
            for k_id in k_id_to_del:
                del _keyfile[k_id]
                del _keys[k_id]
            # check for missing key but tag
            tags_to_del = []
            for it in _tags.items():
                if it[0] not in _keyfile.keys():
                    with codecs.open(self.rpt_key_vs_tags, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m026%(it[1],it[0] ) )
                    taglist.remove(it[1] )
                    idlist.remove(it[0] )
                    tags_to_del.append(it)
            for it in tags_to_del:
                del _tags[it[0] ]
                del _ids[it[1] ]
        except IOError as e:
            if (not self.aprt.is_verbose() ) and (e[0] == 13): pass
            else: raise

        self.keys = _keys
        self.tags = _tags
        self.ids =  _ids

        # Calculate offsets
        self.offset_from_tag = {}
        self.offset_from_id = {}
        for i,n in zip(idlist,range(len(idlist) ) ):
            if n%2: # if n is odd number
                offst = self.aprt.get_param("sched_offset")*(n+1)/2
            else:   # if n is even number
                offst = -self.aprt.get_param("sched_offset")*n/2
            self.offset_from_id[i] = offst
            self.offset_from_tag[_tags[i] ] = offst

        self.sched = {}
        self.txt_sched = {}

        # Schedule
        try:
            with codecs.open(self.rpt_sched_error, "w", encoding='utf-8', errors='backslashreplace') as f:
                f.write(UTF8BOM)    # Clear report file
                f.write(REPORT_TITLE)
            with codecs.open(os.path.join(CAIR_ROOT,"schedule.conf"),"r", encoding='utf-8', errors='backslashreplace') as f:
                sched_data = f.read()
                if sched_data[0] == UTF8BOM: sched_data = sched_data[1:]
            sched_lines = [i.split('#')[0].strip() for i in sched_data.splitlines() ]
            for l,num in zip(sched_lines,range(len(sched_lines) ) ):
                if not l or l == UTF8BOM: continue
                if VERBOSE: print("%3d: %s"%(num+1,l) )
                tokens = [i.strip() for i in l.split(',') ]
                tag = tokens[0]

                # check for tag
                if tag != "*" and tag not in _ids.keys():
                    with codecs.open(self.rpt_sched_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write(T.m027%(num+1,tag) )
                    continue
                # other validity
                ok,retval = valid_sched(tokens[1:],self.aprt.get_param("sched30") )
                if not ok:
                    with codecs.open(self.rpt_sched_error, "a", encoding='utf-8', errors='backslashreplace') as f:
                        f.write("%d, %s\n"%(num+1,retval) )
                    continue

                if VERBOSE:
                    print(tag, "(default)" if tag == "*" else _ids[tag],end=" ")
                    print(repr(retval) )
                if tag != "*":
                    sched_id = self.get_id(tag)
                    self.txt_sched[sched_id] = l
                    self.sched[sched_id] = retval
                else:
                    self.txt_def_sched = l
                    self.def_sched = retval
            if self.def_sched is None:
                raise ValueError("No default schedule")
        except IOError as e:
            if (not self.aprt.is_verbose() ) and (e[0] == 13): pass
            else: raise
        self.last_conf_load = time.time()

    def get_overdue_by_tag(self,tag):
        sched_num = self.stat_dict["data"][tag]["next"]["schd_num"]
        now = round(time.time() )
        if sched_num > 0:
            dsec = now - sched_num
            return dsec
        return None

    def get_overdue(self):  # return missing measurements
        out = []
        tags = [i for i in self.stat_dict["data"].keys() ]
        tags.sort()
        for i in tags:
            dsec = self.get_overdue_by_tag(i)
            if dsec is not None:
                prev = self.prev_dsec.get(i,0)
                if ((i not in self.reported_missing and dsec > 180)  # delta_sec > 180 [s] (3 min)
                    or (dsec - prev) > 3600):   # evely 3600 [s] (1 hour)
                    self.prev_dsec[i] = dsec
                    self.reported_missing.append(i)
                    out.append([i,self.stat_dict["data"][i]["id"],dsec] )
        return out

    def get_key(self,id):
        return self.keys.get(id)
    def get_tag(self,id):
        return self.tags.get(id)
    def get_id(self,tag):
        return self.ids.get(tag)
    def get_sched_by_id(self,id,now):
        l = self.sched.get(id)
        o = self.offset_from_id[id]
        if l is None:
            # print("Default schedule used")
            l = self.def_sched
        return parse_sched_line(l,now,o)
    def get_sched_by_tag(self,tag,now):
        return self.get_sched_by_id(self.get_id(tag),now)
    def get_lasttt_by_id(self,id):
        return self.get_lasttt_by_tag(self.get_tag(id) )
    def get_lasttt_by_tag(self,tag):
        dd = self.stat_dict["data"].get(tag)
        return (dd["last"]["ts_num"],dd["last"]["schd_num"] )
    def is_cont_by_id(self,id):
        if id not in self.sched.keys():
            return False,3.6
        return self.sched[id].get("cont_flag"),self.sched[id].get("cont_thrsh")
    def is_cont_by_tag(self,tag):
        return self.is_cont_by_id(self.get_id(tag) )
    def clear_cont_by_id(self,id):
        if id not in self.sched.keys(): return
        self.sched[id]["cont_flag"] = False
    def clear_cont_by_tag(self,tag):
        self.clear_cont_by_id(self.get_id(tag) )

class app_root_class(object):
    def __init__(self):
        self._params_dict = {}
        # Acceptable Parameter Keys and theif Conversion Functions
        as_is = lambda x: x
        to_int = lambda x: int(x)
        to_flt = lambda x: float(x)
        def valid_data_sync(x):
            try:
                y = int(x)
                print("\n" + esc_blfg + T.m028 + esc_norm + "\n")
                return "off" if y == 0 else "on"
            except ValueError:
                y = x.lower()
                return y if y in "on off off_app_only".split() else None
        def valid_summary_csv(x):
            print("\n" + esc_blfg + T.m029 + esc_norm + "\n")
            return True if x.lower() == "yes" else None
        acceptable_param = {
            "app_root": as_is,
            "data_gen": to_int,
            # "data_sync": to_int,
            "data_sync": valid_data_sync,
            # "file_type": lambda x: "cna" if x.lower() == "cna" else None,
            "file_type": lambda x: x.lower() if x.lower() in "cna csv summary_csv summary_csv+cna".split() else None,
            # "summary_csv": lambda x: True if x.lower() == "yes" else None,
            "summary_csv": valid_summary_csv,
            "sched30": lambda x: False if x.lower() == "disable" else None,
            "sched_offset": lambda x: int(x) if int(x) >= 0 and int(x) <= 60 else None,
            "k_clk_dly": to_flt,
            "use_init_us": to_int,
            "awth_min_sec": to_int,
            "awth_max_sec": to_int,
            "awth_intvl_ratio": to_flt,
            "clk_adj_max": to_flt,
            "clk_adj_min": to_flt,
            "conf_reload": lambda x: [i.strip() for i in x.split() ],
            "verbose": lambda x: True if x.lower() == "yes" else None,
            "conlog": lambda x: True if x.lower() == "yes" else None,
            "tdump": lambda x: True if x.lower() == "yes" else None,
            "lowbatt": to_flt,
            "save_all": lambda x: True if x.lower() == "yes" else None,
            "clkcal_minute": lambda x: int(x) if int(x)>0 and int(x)<=1440 else None,
        }
        # Required keys
        # keys_require = "APP_ROOT DATA_GEN DATA_SYNC".lower().split()
        keys_require = "APP_ROOT DATA_SYNC".lower().split()
        # Default values
        self._param_default = {
            "data_sync": "on",
            "file_type": "csv",
            "summary_csv": False,
            "sched30": True,
            "sched_offset": 10,
            "k_clk_dly": -1.0,
            "use_init_us": 1,
            "awth_min_sec": 60,         # 1 minute
            "awth_max_sec": 1*3600,     # 1 hours
            "awth_intvl_ratio": 0.1,    # 10% of previous actual interval
            "clk_adj_max": 1.10,
            "clk_adj_min": 0.90,
            "conf_reload": "15 45".split(),
            "verbose": False,
            "conlog": False,
            "tdump": False,
            "lowbatt": 2.55,
            "save_all": False,
            "clkcal_minute": 3,
        }

        conf = os.path.join(CAIR_ROOT,"app_root.conf")
        conf2 = os.path.join(CAIR_ROOT,PRIV_DIR,"debug.conf")
        with open(conf, "rb") as f:
            data = f.read()
        if os.path.exists(conf2):
            with open(conf2, "rb") as f:
                data += f.read()

        raw_lines = [i.decode("utf-8",'backslashreplace') for i in data.splitlines() ]
        lines = [i.split("#")[0].strip() for i in raw_lines if i.split("#")[0].strip() ]
        for i in lines:
            tokens = i.split(',')
            if len(tokens) < 2:
                raise ValueError("Not enough items which should be 2: %s"%i)
            key = tokens[0].strip().lower()
            val = tokens[1].strip()

            # Build parameter dictionary
            if key not in acceptable_param.keys():
                raise ValueError("Invalid parameter %s"%i)
            param_val = acceptable_param[key](val)
            if param_val is None:
                raise ValueError("Invalid value for the parameter %s"%i)
            self._params_dict[key] = param_val

        VERBOSE = self._params_dict.get("verbose")

        # Validate parameter dictionary
        if VERBOSE: print("List of parameters")
        keys = [i for i in acceptable_param.keys() ]
        keys.sort()
        for k in keys:
            if VERBOSE: print("\t%-17s"%k,end=" ")
            test = self.get_param(k)
            if VERBOSE: print(test)
        if VERBOSE and self._params_dict.get("conf_reload") and (not self.get_param("sched30") ):
            print(esc_blfg + T.m030 + esc_norm + "\n")
        if self._params_dict.get("summary_csv") and (self.get_param("file_type")!="csv"):
            print(esc_blfg + T.m031 + esc_norm + "\n")

        # Create APP_ROOT if not exists
        apprt = os.path.abspath(self.get_param("app_root") )
        if not os.path.exists(apprt):
            os.makedirs(apprt)

        # Show Special Values
        special_msg = ""
        if self.get_param("file_type") == "cna":
            special_msg += esc_grbg+"** Special Setting: file_type cna **"+esc_norm+"\n"
        if self.get_param("summary_csv"):
            special_msg += esc_grbg+"** Special Setting: summary_csv yes **"+esc_norm+"\n"
        if self.get_param("sched30") == "disable":
            special_msg += esc_blfg+"** Test Setting: sched30 disabled **"+esc_norm+"\n"
        if self.is_verbose():
            special_msg += esc_blfg+"** Test Setting: sched_offset %d **"%self.get_param("sched_offset")+esc_norm+"\n"
        if special_msg:
            print("\n"+special_msg,end=" ")
    def get_pdict(self):
        return (self._params_dict,self._param_default)
    def get_param(self,name):
        value = self._params_dict.get(name.lower() )
        value = self._param_default.get(name.lower() ) if value is None else value
        if value is None:
            raise KeyError("No parameter available for the name %s"%name)
        return value
    # Short-cut methods
    def is_verbose(self):
        return self.get_param("verbose")
    def get_root(self):
        return self.get_param("app_root")
    def get_gen(self):
        return self.get_param("data_gen")
    def get_sync(self):
        return self.get_param("data_sync")
    def get_conlog(self):
        if self.get_param("conlog"):
            tgt_dir = os.path.join("CAIR_ROOT", PRIV_DIR)
            tgt_path = os.path.join(tgt_dir,"console.log")
            if not os.path.exists(tgt_dir): os.makedirs(tgt_dir)
            return os.path.join(tgt_path)
        return None
    def get_effective_file_type(self):
        ft = self.get_param("file_type")
        sm = self.get_param("summary_csv")
        return "summary_csv" if sm and ft == "csv" else ft

