#!/usr/bin/python
# -*- coding: utf8 -*-

from __future__ import print_function
import numpy as np
from numpy.ctypeslib import ndpointer
import ctypes
import os, platform

if os.name == "nt":
    _DLL_NAME = "EEMA_core64.dll" if platform.architecture()[0]=='64bit' else "EEMA_core32.dll"
else:
    _DLL_NAME = "libEEMA_core.so"
_DLL = ctypes.CDLL(_DLL_NAME)
_C_EEMA = _DLL.EEMA
_C_EEMA.argtypes = [
            ctypes.c_uint32,                                    # N
            ndpointer(ctypes.c_double, flags="C_CONTIGUOUS"),   # x
            ctypes.c_double,                                    # a
            ctypes.c_double,                                    # b
            ctypes.c_double,                                    # ini
        ]
_C_HPF  = _DLL.HPF
_C_HPF.argtypes = [
            ctypes.c_uint32,                                    # N
            ndpointer(ctypes.c_double, flags="C_CONTIGUOUS"),   # x
            ctypes.c_double,                                    # a
            ctypes.c_double,                                    # b
            ctypes.c_double,                                    # ini
        ]

def EEMA(x,a,b,ini):
    c_x = np.array(x,dtype=np.float64)
    c_n = ctypes.c_uint32(c_x.size)
    c_a = ctypes.c_double(a)
    c_b = ctypes.c_double(b)
    c_ini = ctypes.c_double(ini)
    
    rc = _C_EEMA(c_n, c_x, c_a, c_b, c_ini)
    
    return c_x

def HPF(x,a,b,ini):
    c_x = np.array(x,dtype=np.float64)
    c_n = ctypes.c_uint32(c_x.size)
    c_a = ctypes.c_double(a)
    c_b = ctypes.c_double(b)
    c_ini = ctypes.c_double(ini)
    
    rc = _C_HPF(c_n, c_x, c_a, c_b, c_ini)
    
    return c_x

# Test funcitons

def _alpha(fc):
    return 2*np.pi*fc*dt

def _iir_lpf(s,a,b,ini):
    y = ini
    o = []
    for i in s:
        y *= b
        y += a*i
        o.append(y)
    return np.array(o)

def _iir_hpf(s,a,b,ini):
    r = ini
    o = []
    for i in s:
        y = i - r
        r *= b
        r += a*i
        o.append(y)
    return np.array(o)

def _show_result(out1,out2,delta,title):
    f, (ax1, ax2) = pl.subplots(2, 1, sharex=True)
    ax1.plot(t,sig,label="Input")
    ax1.plot(t,out1,'.',label="Output 1")
    ax1.plot(t,out2,'.',label="Output 2")
    ax1.grid(True)
    leg = ax1.legend(loc="best")
    leg.draggable()
    ax2.plot(t,out1-out2,'.',label="[Output 1]-[Output 2]")
    ax2.set_ylim(-1e-3,1e-3)
    ax2.grid(True)
    leg = ax2.legend(loc="best")
    leg.draggable()
    ax1.set_title(title)
    pl.show()

if __name__ == "__main__":
    import numpy.random as nprand
    if os.name == 'nt':
        import matplotlib.pyplot as pl
    import timeit

    period = 1
    dt = 1e-3
    f = 10
    N = int(period/dt)
    t = dt*np.arange(N)
    sig = np.sin(2*np.pi*f*t) + 0.1*nprand.randn(N)

    fc = 20
    a = _alpha(fc)
    b = 1 - a

    out1 = _iir_lpf(sig,a,b,0.5)
    out2 = EEMA(sig,a,b,0.5)
    delta = out1 - out2
    t1 = timeit.timeit('_iir_lpf(sig,a,b,0.5)',setup='from __main__ import _iir_lpf,sig,a,b', number=1000)
    t2 = timeit.timeit('EEMA(sig,a,b,0.5)',setup='from __main__ import EEMA,sig,a,b', number=1000)
    print("LPF/EEMA")
    print("time: %.6f, %.6f"%(t1, t2) )
    print("delta: %e - %e"%(delta.min(),delta.max()) )
    if os.name == 'nt':
        _show_result(out1,out2,delta,"LPF/EEMA")

    out1 = _iir_hpf(sig,a,b,0.5)
    out2 = HPF(sig,a,b,0.5)
    delta = out1 - out2
    t1 = timeit.timeit('_iir_hpf(sig,a,b,0.5)',setup='from __main__ import _iir_hpf,sig,a,b', number=1000)
    t2 = timeit.timeit('HPF(sig,a,b,0.5)',setup='from __main__ import HPF,sig,a,b', number=1000)
    print("HPF")
    print("time: %.6f, %.6f"%(t1, t2) )
    print("delta: %e - %e"%(delta.min(),delta.max()) )
    if os.name == 'nt':
        _show_result(out1,out2,delta,"HPF")
